<?php

namespace App\Models;

use Spatie\Permission\Guard;
use Illuminate\Database\Eloquent\Model;
use Spatie\Permission\Traits\HasPermissions;
use Spatie\Permission\Exceptions\RoleDoesNotExist;
use Spatie\Permission\Exceptions\GuardDoesNotMatch;
use Spatie\Permission\Exceptions\RoleAlreadyExists;
use Spatie\Permission\Contracts\Role as RoleContract;
use Spatie\Permission\Traits\RefreshesPermissionCache;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Activitylog\Traits\LogsActivity;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Schema;

class Role extends BaseModel implements RoleContract
{
    use HasPermissions;
    use RefreshesPermissionCache;

    protected $guarded = ['id'];
    // Customize //
    use SoftDeletes;
    protected static $logAttributes = ['name', 'guard_name'];
    protected $fillable = ['name', 'guard_name'];
    public $searchable = ['name', 'guard_name'];
    public $default_sort = 'id';
    public $default_sort_order = 'desc';
    public $model = 'role';
    public $fields_array = array(
        'id' => array(
            'field_name' => 'id',
            'db_name' => 'id',
            'type' => 'text',
            'placeholder' => 'Id',
            'listing' => true,
            'show_in_form' => false,
            'sort' => true,
            'required' => false,
            'value' => '',
            'width' => '50'
        ),
        'name' => array(
            'field_name' => 'name',
            'db_name' => 'name',
            'type' => 'text',
            'placeholder' => 'Name',
            'listing' => true,
            'sort' => true,
            'required' => true,
            'value' => '',
            'width' => '50'
        ),
        'guard_name' => array(
            'field_name' => 'guard_name',
            'db_name' => 'guard_name',
            'type' => 'text',
            'placeholder' => 'Guard Name',
            'listing' => true,
            'sort' => true,
            'required' => false,
            'value' => '',
            'width' => '50',
            'show_in_form' => true
        )
    );
    // END //
    public function __construct(array $attributes = [])
    {
        $attributes['guard_name'] = $attributes['guard_name'] ?? config('auth.defaults.guard');

        parent::__construct($attributes);

        $this->setTable(config('permission.table_names.roles'));
    }

    public static function create(array $attributes = [])
    {
        $attributes['guard_name'] = $attributes['guard_name'] ?? Guard::getDefaultName(static::class);

        if (static::where('name', $attributes['name'])->where('guard_name', $attributes['guard_name'])->first()) {
            throw RoleAlreadyExists::create($attributes['name'], $attributes['guard_name']);
        }

        return static::query()->create($attributes);
    }

    /**
     * A role may be given various permissions.
     */
    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(
            config('permission.models.permission'),
            config('permission.table_names.role_has_permissions'),
            'role_id',
            'permission_id'
        );
    }

    /**
     * A role belongs to some users of the model associated with its guard.
     */
    public function users(): MorphToMany
    {
        return $this->morphedByMany(
            getModelForGuard($this->attributes['guard_name']),
            'model',
            config('permission.table_names.model_has_roles'),
            'role_id',
            config('permission.column_names.model_morph_key')
        );
    }

    /**
     * Find a role by its name and guard name.
     *
     * @param string $name
     * @param string|null $guardName
     *
     * @return \Spatie\Permission\Contracts\Role|\Spatie\Permission\Models\Role
     *
     * @throws \Spatie\Permission\Exceptions\RoleDoesNotExist
     */
    public static function findByName(string $name, $guardName = null): RoleContract
    {
        $guardName = $guardName ?? Guard::getDefaultName(static::class);

        $role = static::where('name', $name)->where('guard_name', $guardName)->first();

        if (! $role) {
            throw RoleDoesNotExist::named($name);
        }

        return $role;
    }

    public static function findById(int $id, $guardName = null): RoleContract
    {
        $guardName = $guardName ?? Guard::getDefaultName(static::class);

        $role = static::where('id', $id)->where('guard_name', $guardName)->first();

        if (! $role) {
            throw RoleDoesNotExist::withId($id);
        }

        return $role;
    }

    /**
     * Find or create role by its name (and optionally guardName).
     *
     * @param string $name
     * @param string|null $guardName
     *
     * @return \Spatie\Permission\Contracts\Role
     */
    public static function findOrCreate(string $name, $guardName = null): RoleContract
    {
        $guardName = $guardName ?? Guard::getDefaultName(static::class);

        $role = static::where('name', $name)->where('guard_name', $guardName)->first();

        if (! $role) {
            return static::query()->create(['name' => $name, 'guard_name' => $guardName]);
        }

        return $role;
    }

    /**
     * Determine if the user may perform the given permission.
     *
     * @param string|Permission $permission
     *
     * @return bool
     *
     * @throws \Spatie\Permission\Exceptions\GuardDoesNotMatch
     */
    public function hasPermissionTo($permission): bool
    {
        $permissionClass = $this->getPermissionClass();

        if (is_string($permission)) {
            $permission = $permissionClass->findByName($permission, $this->getDefaultGuardName());
        }

        if (is_int($permission)) {
            $permission = $permissionClass->findById($permission, $this->getDefaultGuardName());
        }

        if (! $this->getGuardNames()->contains($permission->guard_name)) {
            throw GuardDoesNotMatch::create($permission->guard_name, $this->getGuardNames());
        }

        return $this->permissions->contains('id', $permission->id);
    }
    
    public function view(Request $request) {
        $permissions = '';
        $result = $this->getModel($request->id);
        $permissions = \App\Models\Role::findByName($result->name)->permissions;
        $modules = \App\Models\Module::all()->where('deleted_at', null);
        $actions = \App\Models\Action::all()->where('deleted_at', null);
        return array('result' => $result, 'modules' => $modules, 'actions' => $actions, 'permissions' => $permissions);
    }
    
    public function getListQuery(Request $request){
        $sort = $request->sort ? $request->sort : $this->getModel()->default_sort;
        $sort_order = $request->sort_order ? $request->sort_order : $this->getModel()->default_sort_order;
        $table = $this->getModel()->getTable();
        $raw_sql = "$table.*";
        if(!empty($this->select_fields)){
            $raw_sql = implode(',',$this->select_fields);
        }
        
//        if(!empty($this->join_colums)){
//            $raw_sql .= implode($this->join_colums);
//        }
        
        $query = DB::table($table)
                    ->select(DB::raw($raw_sql));

        if ($request->user()->company_id != 2) {
            $query->whereNotIn('id', [7,5]);
        }
        
        // Filter records using where clause
        $where_clause = $request->where_clause ? $request->where_clause : '';
        if ($where_clause != '') {
            $where_clause = json_decode($where_clause, 1);
            $where_fields = $where_clause['where_fields'];
            $where_values = $where_clause['where_values'];

            foreach ($where_fields as $key => $where_field) {
                $query->where($where_field, $where_values[$key]);
            }
        }
        
        if (!empty($this->searchable) && $request->search != '') {
            $concat_fields = 'concat(';
            foreach ($this->searchable as $field) {
                $concat_fields .= 'COALESCE(' . $table . '.' . $field . ", ''),'',";
            }
            $concat_fields = rtrim($concat_fields, ",'',");
            $concat_fields .= ')';
            $query->where([[DB::raw($concat_fields), 'like', "%$request->search%"]]);
        }
        $query->where('deleted_at', null);
        $query->orderBy($sort, $sort_order);
        
        return $query;
    }
}
